#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/event_groups.h"
#include "esp_system.h"
#include "esp_wifi.h"
#include "esp_event.h"
#include "esp_log.h"
#include "nvs_flash.h"
#include "lwip/err.h"
#include "lwip/sockets.h"
#include "lwip/sys.h"
#include "lwip/netdb.h"
#include "lwip/dns.h"
#include "sdkconfig.h"

#include "pins.h"
#include "sdcardspi.h"

#include "task_wifi.h"

#define WIFI_SSID_SIZE 32
#define WIFI_PASS_SIZE 64
#define RTC_UPDATE_TIME 600


#define EXAMPLE_ESP_MAXIMUM_RETRY 3

#define ESP_WIFI_SAE_MODE WPA3_SAE_PWE_HUNT_AND_PECK
#define EXAMPLE_H2E_IDENTIFIER ""

#define ESP_WIFI_SCAN_AUTH_MODE_THRESHOLD WIFI_AUTH_WPA_WPA2_PSK


#define WIFI_CONNECTED_BIT BIT0
#define WIFI_FAIL_BIT      BIT1

/* Constants that aren't configurable in menuconfig */
#define WEB_SERVER "timehttp.bp4k.net"
#define WEB_PORT "80"
#define WEB_PATH "/?epoch=1"

static const char *TAG = "task_wifi";

static const char *REQUEST = "GET " WEB_PATH " HTTP/1.0\r\n"
    "Host: "WEB_SERVER":"WEB_PORT"\r\n"
    "User-Agent: esp-idf/1.0 esp32\r\n"
    "\r\n";

static int s_retry_num = 0;

/* FreeRTOS event group to signal when we are connected*/
static EventGroupHandle_t s_wifi_event_group;

/* local fns */
int get_wifi_creds(char *username, size_t usz, char *password, size_t psz);
void wifi_init_sta(char *username, char *password);
void http_get_task(void);


void task_wifi_time(void *pvParams) {
    char wifi_ssid[WIFI_SSID_SIZE];
    char wifi_pass[WIFI_PASS_SIZE];

    /* simply get the SSID and passwd */
    if (!gt_global_sdsupport) {
        ESP_EARLY_LOGE(TAG, "Cannot obtain wifi password since no sd support");
        vTaskDelete(NULL);
    }

    if (get_wifi_creds(wifi_ssid, WIFI_SSID_SIZE,
                wifi_pass, WIFI_PASS_SIZE))
        vTaskDelete(NULL);

    ESP_EARLY_LOGI(TAG, "Wi-Fi SSID: <<%s>>", wifi_ssid);
    ESP_EARLY_LOGI(TAG, "Wi-Fi Password: <<%s>>", wifi_pass);


    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
      ESP_ERROR_CHECK(nvs_flash_erase());
      ret = nvs_flash_init();
    }
    ESP_ERROR_CHECK(ret);

    if (CONFIG_LOG_MAXIMUM_LEVEL > CONFIG_LOG_DEFAULT_LEVEL) {
        /* If you only want to open more logs in the wifi module, you need to make the max level greater than the default level,
         * and call esp_log_level_set() before esp_wifi_init() to improve the log level of the wifi module. */
        //esp_log_level_set("wifi", CONFIG_LOG_MAXIMUM_LEVEL);
    }

    ESP_LOGI(TAG, "ESP_WIFI_MODE_STA");
    wifi_init_sta(wifi_ssid, wifi_pass);
    http_get_task();

    vTaskDelete(NULL);
}


int get_wifi_creds(char *uname, size_t usz, char *pw, size_t psz) {
    int nzec = 0;
    FILE *fp = fopen(SD_PATH("config/wifi.conf"), "rb");

    int i;

    if (!fp) {
        perror("fopen");
        ESP_EARLY_LOGE(TAG, "cannot obtain wifi ssid and password");
        nzec = 1;
        goto err_no_fclose;
    }

    i = 0;
    while (
            i < usz - 1
            && fread(&uname[i], sizeof(char), 1, fp)
            && uname[i] != '\n')
        ++i;

    if (uname[i] == '\n')
        uname[i] = '\0';
    else {
        ESP_EARLY_LOGE(TAG, "invalid wifi config, endl after ssid missing");
        nzec = 1;
        goto err_fclose;
    }

    i = 0;
    while (
            i < psz - 1
            && fread(&pw[i], sizeof(char), 1, fp)
            && pw[i] != '\n')
        ++i;

    if (pw[i] == '\n')
        pw[i] = '\0';
    else {
        ESP_EARLY_LOGW(TAG, "wifi password might be truncated");
        pw[psz - 1] = '\0';
    }


err_fclose:
    fclose(fp);
err_no_fclose:
    return nzec;
}


static void event_handler(void* arg, esp_event_base_t event_base,
                                int32_t event_id, void* event_data)
{
    if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_START) {
        esp_wifi_connect();
    } else if (event_base == WIFI_EVENT && event_id == WIFI_EVENT_STA_DISCONNECTED) {
        if (s_retry_num < EXAMPLE_ESP_MAXIMUM_RETRY) {
            esp_wifi_connect();
            s_retry_num++;
            ESP_LOGI(TAG, "retry to connect to the AP");
        } else {
            xEventGroupSetBits(s_wifi_event_group, WIFI_FAIL_BIT);
        }
        ESP_LOGI(TAG,"connect to the AP fail");
    } else if (event_base == IP_EVENT && event_id == IP_EVENT_STA_GOT_IP) {
        ip_event_got_ip_t* event = (ip_event_got_ip_t*) event_data;
        ESP_LOGI(TAG, "got ip:" IPSTR, IP2STR(&event->ip_info.ip));
        s_retry_num = 0;
        xEventGroupSetBits(s_wifi_event_group, WIFI_CONNECTED_BIT);
    }
}


void wifi_init_sta(char *username, char *password) {
    s_wifi_event_group = xEventGroupCreate();

    ESP_ERROR_CHECK(esp_netif_init());

    ESP_ERROR_CHECK(esp_event_loop_create_default());
    esp_netif_create_default_wifi_sta();

    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_wifi_init(&cfg));

    esp_event_handler_instance_t instance_any_id;
    esp_event_handler_instance_t instance_got_ip;
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                                                        ESP_EVENT_ANY_ID,
                                                        &event_handler,
                                                        NULL,
                                                        &instance_any_id));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(IP_EVENT,
                                                        IP_EVENT_STA_GOT_IP,
                                                        &event_handler,
                                                        NULL,
                                                        &instance_got_ip));

    wifi_config_t wifi_config = {
        .sta = {
            /*
            .ssid = username,
            .password = password,
            */
            /* Authmode threshold resets to WPA2 as default if password matches WPA2 standards (password len => 8).
             * If you want to connect the device to deprecated WEP/WPA networks, Please set the threshold value
             * to WIFI_AUTH_WEP/WIFI_AUTH_WPA_PSK and set the password with length and format matching to
             * WIFI_AUTH_WEP/WIFI_AUTH_WPA_PSK standards.
             */
            .threshold.authmode = ESP_WIFI_SCAN_AUTH_MODE_THRESHOLD,
            .sae_pwe_h2e = ESP_WIFI_SAE_MODE,
            .sae_h2e_identifier = EXAMPLE_H2E_IDENTIFIER,
        },
    };
    strncpy((char *) wifi_config.sta.ssid, username, 32);
    strncpy((char *) wifi_config.sta.password, password, 64);
    ESP_ERROR_CHECK(esp_wifi_set_mode(WIFI_MODE_STA));
    ESP_ERROR_CHECK(esp_wifi_set_config(WIFI_IF_STA, &wifi_config));
    ESP_ERROR_CHECK(esp_wifi_start() );

    ESP_EARLY_LOGI(TAG, "wifi_init_sta finished.");

    /* Waiting until either the connection is established (WIFI_CONNECTED_BIT) or connection failed for the maximum
     * number of re-tries (WIFI_FAIL_BIT). The bits are set by event_handler() (see above) */
    EventBits_t bits = xEventGroupWaitBits(s_wifi_event_group,
            WIFI_CONNECTED_BIT | WIFI_FAIL_BIT,
            pdFALSE,
            pdFALSE,
            portMAX_DELAY);

    /* xEventGroupWaitBits() returns the bits before the call returned, hence we can test which event actually
     * happened. */
    if (bits & WIFI_CONNECTED_BIT) {
        ESP_EARLY_LOGI(TAG, "connected to ap");
    } else if (bits & WIFI_FAIL_BIT) {
        ESP_EARLY_LOGI(TAG, "Failed to connect to SSID");
    } else {
        ESP_EARLY_LOGE(TAG, "UNEXPECTED EVENT");
    }
}


void http_get_task()
{
    const struct addrinfo hints = {
        .ai_family = AF_INET,
        .ai_socktype = SOCK_STREAM,
    };
    struct addrinfo *res;
    struct in_addr *addr;
    int s, r;
    char recv_buf[64];

    char rnrn[4];
    int8_t dosave = -1;
    char tmbuf[16];

    while(1) {
        int err = getaddrinfo(WEB_SERVER, WEB_PORT, &hints, &res);

        if(err != 0 || res == NULL) {
            ESP_LOGE(TAG, "DNS lookup failed err=%d res=%p", err, res);
            vTaskDelay(1000 / portTICK_PERIOD_MS);
            continue;
        }

        /* Code to print the resolved IP.

           Note: inet_ntoa is non-reentrant, look at ipaddr_ntoa_r for "real" code */
        addr = &((struct sockaddr_in *)res->ai_addr)->sin_addr;
        ESP_LOGI(TAG, "DNS lookup succeeded. IP=%s", inet_ntoa(*addr));

        s = socket(res->ai_family, res->ai_socktype, 0);
        if(s < 0) {
            ESP_LOGE(TAG, "... Failed to allocate socket.");
            freeaddrinfo(res);
            vTaskDelay(1000 / portTICK_PERIOD_MS);
            continue;
        }
        ESP_LOGI(TAG, "... allocated socket");

        if(connect(s, res->ai_addr, res->ai_addrlen) != 0) {
            ESP_LOGE(TAG, "... socket connect failed errno=%d", errno);
            close(s);
            freeaddrinfo(res);
            vTaskDelay(4000 / portTICK_PERIOD_MS);
            continue;
        }

        ESP_LOGI(TAG, "... connected");
        freeaddrinfo(res);

        if (write(s, REQUEST, strlen(REQUEST)) < 0) {
            ESP_LOGE(TAG, "... socket send failed");
            close(s);
            vTaskDelay(4000 / portTICK_PERIOD_MS);
            continue;
        }
        ESP_LOGI(TAG, "... socket send success");

        struct timeval receiving_timeout;
        receiving_timeout.tv_sec = 5;
        receiving_timeout.tv_usec = 0;
        if (setsockopt(s, SOL_SOCKET, SO_RCVTIMEO, &receiving_timeout,
                sizeof(receiving_timeout)) < 0) {
            ESP_LOGE(TAG, "... failed to set socket receiving timeout");
            close(s);
            vTaskDelay(4000 / portTICK_PERIOD_MS);
            continue;
        }
        ESP_LOGI(TAG, "... set socket receiving timeout success");

        /* Read HTTP response */
        do {
            bzero(recv_buf, sizeof(recv_buf));
            r = read(s, recv_buf, sizeof(recv_buf)-1);
            for(int i = 0; i < r; i++) {
                putchar(recv_buf[i]);
                if (dosave)
                rnrn[3] = rnrn[2];
                rnrn[2] = rnrn[1];
                rnrn[1] = rnrn[0];
                rnrn[0] = recv_buf[i];

                if (!strncmp(rnrn, "\r\n", 2)) dosave = 0;

                if (dosave > -1 && dosave < 15) {
                    tmbuf[dosave++] = recv_buf[i];
                }
            }
        } while(r > 0);
        tmbuf[dosave - 1] = '\0';

        ESP_LOGI(TAG, "... done reading from socket. Last read return=%d errno=%d.", r, errno);
        close(s);

        time_t tm = (time_t) atol(tmbuf + 2);
        ESP_EARLY_LOGI(TAG, "the time is: <<%d>>", (int) tm);

        struct timeval timestruct = {
            .tv_sec = tm - 60 * 60 * 5
        };
        if (settimeofday(&timestruct, NULL))
            ESP_EARLY_LOGE(TAG, "failed to update time");

        vTaskDelay(pdMS_TO_TICKS(RTC_UPDATE_TIME * 1000));
        ESP_LOGI(TAG, "Starting again!");
        dosave = -1;
    }
}
