#include <time.h>
#include <math.h>

#include "freertos/FreeRTOS.h"
#include "driver/gpio.h"
#include "driver/uart.h"
#include "iot_button.h"
#include "esp_log.h"

#include "Goldelox_const4D.h"
#include "goldeloxSerial.h"
#include "esp_system.h"

#include "task_lcd.h"
#include "task_button.h"
#include "app_health.h"
#include "pins.h"

static const char* TAG = "task_lcd";
const uart_port_t uart_num = UART_NUM_1;
int g_time = 0;

gl_display_t *error_disp = NULL;

/* list of all apps as function pointers */
struct app_t applist[NUM_APPS] = {
    {
        .fn = &lcd_clock_loop,
        .icon.width = 0,
        .icon.height = 0,
        .icon.data = NULL,
        .name = "Clock"
    },
    {
        .fn = &lcd_boot_flash_app,
        .icon.width = 0,
        .icon.height = 0,
        .icon.data = NULL,
        .name = "About SWOS"
    },
    {
        .fn = &app_health_data,
        .icon.width = 0,
        .icon.height = 0,
        .icon.data = NULL,
        .name = "WatchDog Test"
    },
    {
        .fn = &app_health_data2,
        .icon.width = 0,
        .icon.height = 0,
        .icon.data = NULL,
        .name = "Exercise"
    }
};

/* ================================================================= */
/* functions definitons */

void lcd_clock_draw_face(gl_display_t *);
void lcd_clock_draw_hand(gl_display_t *, uint16_t, uint16_t, uint8_t, gl_word_t);

int write_tx(const void*, size_t);
int read_rx(void *, size_t, uint32_t);
int flush_rx();
int flush_tx(uint32_t);
void faster_baud();
void slower_baud();
void lcd_homepage_draw_base(gl_display_t *, uint8_t, uint8_t, uint8_t);

/* ================================================================= */

int write_tx(const void *data, size_t sz) {
    return uart_write_bytes(uart_num, data, sz);
}


int read_rx(void *data, size_t sz, uint32_t wait_ms) {
    if (sizeof(size_t) > sizeof(uint32_t) && sz > UINT32_MAX)
        return -1;
    return uart_read_bytes(uart_num, data, (uint32_t) sz,
            (TickType_t) wait_ms / portTICK_PERIOD_MS);
}


int flush_rx() {
    return (int) uart_flush(uart_num);
}


int flush_tx(uint32_t wait_ms) {
    return (int) uart_wait_tx_done(uart_num,
            (TickType_t) wait_ms / portTICK_PERIOD_MS);
}


void faster_baud() {
    uart_config_t uart_config = {
        .baud_rate = 600000,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_CTS_RTS,
        .rx_flow_ctrl_thresh = 10,
    };
    // Configure UART parameters
    ESP_ERROR_CHECK(uart_param_config(uart_num, &uart_config));
}


void slower_baud() {
    uart_config_t uart_config = {
        .baud_rate = 9600,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_CTS_RTS,
        .rx_flow_ctrl_thresh = 10,
    };
    // Configure UART parameters
    ESP_ERROR_CHECK(uart_param_config(uart_num, &uart_config));
}

int Callback4DImpl (int, unsigned char) {
    if (error_disp && g_time % 3 == 0) {
        printf("resetting uLCD...\n");
        ESP_ERROR_CHECK(gpio_set_level(PIN_ULCD_RESET, 0));
        vTaskDelay(100 / portTICK_PERIOD_MS);
        ESP_ERROR_CHECK(gpio_set_level(PIN_ULCD_RESET, 1));
        gl_reset2(error_disp);
        //gl_setbaudWait(error_disp, BAUD_600000, &faster_baud);
    }
    if (g_time > 4)
        esp_restart();
    g_time++;
    return 10;
}
/* ================================================================= */

void lcd_task(void) {
    // Setup UART buffered IO with event queue
    const int uart_buffer_size = UART_HW_FIFO_LEN(uart_num) + 4;
    QueueHandle_t uart_queue;

    Callback4D = &Callback4DImpl;

    gl_serif_t serif_ulcd = {
        .write_tx = write_tx,
        .read_rx = read_rx,
        .flush_rx = flush_rx,
        .flush_tx = flush_tx
    };

    gl_display_t disp;

    ESP_ERROR_CHECK(gpio_reset_pin(PIN_ULCD_RESET));
    ESP_ERROR_CHECK(gpio_set_direction(PIN_ULCD_RESET, GPIO_MODE_OUTPUT));
    ESP_ERROR_CHECK(gpio_set_drive_capability(PIN_ULCD_RESET, GPIO_DRIVE_CAP_1));

    ESP_ERROR_CHECK(uart_driver_install(uart_num, uart_buffer_size, uart_buffer_size, 10, &uart_queue, 0));

    uart_config_t uart_config = {
        .baud_rate = 9600,
        .data_bits = UART_DATA_8_BITS,
        .parity = UART_PARITY_DISABLE,
        .stop_bits = UART_STOP_BITS_1,
        .flow_ctrl = UART_HW_FLOWCTRL_CTS_RTS,
        .rx_flow_ctrl_thresh = 10,
    };
    // Configure UART parameters
    ESP_ERROR_CHECK(uart_param_config(uart_num, &uart_config));
    // tx then rx
    ESP_ERROR_CHECK(uart_set_pin(uart_num, PIN_ULCD_TX, PIN_ULCD_RX,
                UART_PIN_NO_CHANGE, UART_PIN_NO_CHANGE));

    ESP_ERROR_CHECK(gpio_set_level(PIN_ULCD_RESET, 0));
    vTaskDelay(100 / portTICK_PERIOD_MS);
    ESP_ERROR_CHECK(gpio_set_level(PIN_ULCD_RESET, 1));
    printf("Hello world!\n");

    gl_init(&disp, &serif_ulcd);

    //error_disp = &disp;

    /* change baud rate */
    gl_setbaudWait(&disp, BAUD_600000, &faster_baud);
    lcd_boot_flash(&disp, 1);
    //lcd_clock_loop(&disp);
    lcd_homepage_loop(&disp);
}


void lcd_boot_flash(gl_display_t *disp, uint8_t ret) {
    gl_gfx_BGcolour(disp, WHITE);
    gl_gfx_Cls(disp);

    /* draw straps */
    for (uint8_t r = 0; r <= LCD_HEIGHT; r+= 4) {
        gl_gfx_RectangleFilled(disp,
                LCD_WIDTH / 2 - BOOT_STRAP_WIDTH / 2 - BOOT_CIRCL_OFFSET,
                LCD_HEIGHT / 2 - r / 2,
                LCD_WIDTH / 2 + BOOT_STRAP_WIDTH / 2 - BOOT_CIRCL_OFFSET,
                LCD_HEIGHT / 2 + r / 2, BLACK);
        vTaskDelay(pdMS_TO_TICKS(10));
    }

    /* circle grows */
    for (uint8_t r = 1; r <= BOOT_CIRCL_RAD; r += 1) {
        gl_gfx_CircleFilled(disp,
                LCD_WIDTH / 2 - BOOT_CIRCL_OFFSET,
                LCD_HEIGHT / 2 , r,
                TOMATO);
        vTaskDelay(pdMS_TO_TICKS(15));
    }


    gl_txt_FGcolour(disp, BLACK);
    gl_txt_BGcolour(disp, WHITE);
    gl_txt_MoveCursor(disp, 6, 10);
    gl_txt_Bold(disp, 1);
    gl_putstr(disp, (uint8_t *) "SMART");
    gl_txt_MoveCursor(disp, 7, 10);
    gl_txt_Bold(disp, 1);
    gl_putstr(disp, (uint8_t *) "WATCH");
    gl_txt_MoveCursor(disp, 8, 10);
    gl_txt_Bold(disp, 1);
    gl_putstr(disp, (uint8_t *) "OS v0.1");
    gl_txt_MoveCursor(disp, 10, 10);
    gl_putstr(disp, (uint8_t *) "Anish &");
    gl_txt_MoveCursor(disp, 11, 10);
    gl_putstr(disp, (uint8_t *) "Advaith");

    if (ret) {
        ESP_ERROR_CHECK(gpio_set_level(PIN_VIBRATOR, 1));
        vTaskDelay(pdMS_TO_TICKS(500));
        ESP_ERROR_CHECK(gpio_set_level(PIN_VIBRATOR, 0));
        vTaskDelay(pdMS_TO_TICKS(1500));
        return;
    }


    while(1) {
        struct gt_btn_msg_t *btnmsg;
        while (xQueueReceive(gt_btnq, (void *) &btnmsg, portMAX_DELAY)
                == pdPASS) {
            if (btnmsg->btn == GT_BTN_WHITE && btnmsg->evt
                    == BUTTON_SINGLE_CLICK) {
                return;
            } else if (btnmsg->btn == GT_BTN_RED && btnmsg->evt
                    == BUTTON_DOUBLE_CLICK) {
                ESP_LOGI(TAG, "Writing diego swears lol");
                gl_txt_BGcolour(disp, WHITE);
                gl_txt_FGcolour(disp, GREEN);
                gl_txt_MoveCursor(disp, 0, 0);
                gl_putstr(disp, (uint8_t *)
                        "Diego Swears: 2");
            }
        }
    }
}


void lcd_boot_flash_app(gl_display_t *disp) {
    lcd_boot_flash(disp, 0);
}


void lcd_clock_loop(gl_display_t *disp) {
    gl_gfx_BGcolour(disp, BLACK);
    gl_gfx_Cls(disp);
    lcd_clock_draw_face(disp);
    gl_txt_FGcolour(disp, DARKGREEN);
    gl_txt_BGcolour(disp, ALICEBLUE);
    while (1) {
        struct gt_btn_msg_t *btnmsg;
        time_t tm = time(NULL);
        struct tm *timest = localtime(&tm);
        char txt[12];
        // snprintf(txt, 10, "%lld", (long long) tm);
        snprintf(txt, 12, "%02d:%02d %s",
                timest->tm_hour % 12 == 0 ? 12 : timest->tm_hour % 12,
                timest->tm_min,
                (timest->tm_hour > 12) ? "PM" : "AM");
        /*gl_txt_Italic(disp, 1);*/
        uint16_t chw = gl_txt_strw(disp, txt);
        gl_gfx_MoveTo(disp, (LCD_WIDTH / 2) - (chw / 2), (LCD_HEIGHT / 2)
                + (CLOCKSCR_CIRCRAD / 2));
        gl_putstr(disp, (unsigned char *) txt);
        // gl_txt_MoveCursor(disp, 1, 0);
        // gl_putstr(disp, (uint8_t *) txt);
        lcd_clock_draw_hand(disp, (tm + 60 - 1) % 60,
                60, CLOCKSCR_TIMEHDSE, ALICEBLUE);
        lcd_clock_draw_hand(disp, (tm + (60 * 60) - 1) % (60 * 60),
                (60 * 60), CLOCKSCR_TIMEHDMN,
                ALICEBLUE);
        lcd_clock_draw_hand(disp, (tm + (60 * 60 * 12) - 1) % (60 * 60 * 12),
                (60 * 60 * 12), CLOCKSCR_TIMEHDHR, ALICEBLUE);
        lcd_clock_draw_hand(disp, tm % 60, 60, CLOCKSCR_TIMEHDSE, RED);
        lcd_clock_draw_hand(disp, tm % (60 * 60), (60 * 60), CLOCKSCR_TIMEHDMN,
                BLACK);
        lcd_clock_draw_hand(disp, tm % (60 * 60 * 12),
                (60 * 60 * 12), CLOCKSCR_TIMEHDHR, BLACK);
        while (xQueueReceive(gt_btnq, (void *) &btnmsg, 0) == pdPASS) {
            if (btnmsg->btn == GT_BTN_WHITE
                    && btnmsg->evt == BUTTON_SINGLE_CLICK) {
                return;
            }
        }
        vTaskDelay(pdMS_TO_TICKS(500));
    }
}


void lcd_homepage_draw_face(gl_display_t *disp, uint8_t num_rows,
        uint8_t num_cols, uint8_t num_padd) {
    gl_gfx_BGcolour(disp, BLACK);
    gl_gfx_Cls(disp);

    for (uint8_t i = 0; i < num_rows; ++i) {
        for (uint8_t j = 0; j < num_cols; ++j) {
            gl_gfx_RectangleFilled(disp,
                    APPSCR_TOPCALCVW(num_cols, num_padd, j + 1),
                    APPSCR_TOPCALCVH(num_rows, num_padd, i + 1),
                    APPSCR_TOPCALCVW(num_cols, num_padd, j + 1)
                        + APPSCR_PADCALCVW(num_cols, num_padd),
                    APPSCR_TOPCALCVH(num_rows, num_padd, i + 1)
                        + APPSCR_PADCALCVH(num_rows, num_padd),
                    WHITE);
        }
    }
}


void lcd_homepage_draw_color(gl_display_t *disp, uint8_t num_rows,
        uint8_t num_cols, uint8_t num_padd, uint8_t old_r, uint8_t old_c,
        uint8_t new_r, uint8_t new_c) {
    gl_gfx_RectangleFilled(disp,
            APPSCR_TOPCALCVW(num_cols, num_padd, old_c + 1),
            APPSCR_TOPCALCVH(num_rows, num_padd, old_r + 1),
            APPSCR_TOPCALCVW(num_cols, num_padd, old_c + 1)
                + APPSCR_PADCALCVW(num_cols, num_padd),
            APPSCR_TOPCALCVH(num_rows, num_padd, old_r + 1)
                + APPSCR_PADCALCVH(num_rows, num_padd),
            WHITE);
    gl_gfx_RectangleFilled(disp,
            APPSCR_TOPCALCVW(num_cols, num_padd, new_c + 1),
            APPSCR_TOPCALCVH(num_rows, num_padd, new_r + 1),
            APPSCR_TOPCALCVW(num_cols, num_padd, new_c + 1)
                + APPSCR_PADCALCVW(num_cols, num_padd),
            APPSCR_TOPCALCVH(num_rows, num_padd, new_r + 1)
                + APPSCR_PADCALCVH(num_rows, num_padd),
            GREEN);
}


void lcd_homepage_loop(gl_display_t *disp) {
    uint8_t num_rows = 2;
    uint8_t num_cols = 2;
    uint8_t num_padd = APPSCR_PAD;
    uint8_t curr_idx = num_rows * num_cols - 1;
    lcd_clock_loop(disp);
    lcd_homepage_draw_face(disp, num_rows, num_cols, num_padd);

    while(1) {
        struct gt_btn_msg_t *btnmsg;
        while (xQueueReceive(gt_btnq, (void *) &btnmsg, portMAX_DELAY)
                == pdPASS) {
            uint16_t chw;
            char *txt;
            uint8_t old_idx = curr_idx;
            if (btnmsg->btn == GT_BTN_YELLOW
                    && btnmsg->evt == BUTTON_SINGLE_CLICK) {
                gl_gfx_Cls(disp);
                applist[curr_idx].fn(disp);
                lcd_homepage_draw_face(disp, num_rows, num_cols, num_padd);
                continue;
            } else if (btnmsg->btn == GT_BTN_BLUE) {
                if (btnmsg->evt == BUTTON_SINGLE_CLICK) {
                    curr_idx = (curr_idx + 1) % (NUM_APPS);
                } else if (btnmsg->evt == BUTTON_DOUBLE_CLICK) {
                    curr_idx = (curr_idx + num_rows) % (NUM_APPS);
                }
            } else if (btnmsg->btn == GT_BTN_GREEN) {
                if (btnmsg->evt == BUTTON_SINGLE_CLICK) {
                    curr_idx = (curr_idx + NUM_APPS - 1)
                        % (NUM_APPS);
                } else if (btnmsg->evt == BUTTON_DOUBLE_CLICK) {
                    curr_idx = (curr_idx + NUM_APPS - num_rows)
                        % (NUM_APPS);
                }
            }
            lcd_homepage_draw_color(disp, num_rows, num_cols, num_padd,
                    old_idx / num_rows, old_idx % num_rows,
                    curr_idx / num_rows, curr_idx % num_rows);
            /* draw app name */
            txt = applist[curr_idx].name;
            gl_gfx_RectangleFilled(disp, 0, LCD_HHEIGHT, LCD_WIDTH, LCD_HEIGHT,
                    BLACK);
            chw = gl_txt_strw(disp, txt);
            gl_gfx_MoveTo(disp, LCD_WIDTH / 2 - chw / 2, LCD_HHEIGHT
                    + APPSCR_PAD);
            gl_txt_BGcolour(disp, BLACK);
            gl_txt_FGcolour(disp, WHITE);
            gl_putstr(disp, (uint8_t *) txt);
        }
    }
}


void lcd_clock_draw_face(gl_display_t *disp) {
    /* draw the circle */
    gl_gfx_CircleFilled(disp, LCD_WIDTH / 2, LCD_HEIGHT / 2, CLOCKSCR_CIRCRAD,
            ALICEBLUE);

    for (double angle = 0; angle < 2 * M_PI; angle += (2 * M_PI) / 12) {
        /* draw hour ticks */

        /* full 'x' tick start length */
        int8_t fxs = cos(angle) * (CLOCKSCR_CIRCRAD - (CLOCKSCR_TIMEMKHR / 2));
        /* full 'y' tick start length */
        int8_t fys = sin(angle) * (CLOCKSCR_CIRCRAD - (CLOCKSCR_TIMEMKHR / 2));

        /* full 'x' tick end length */
        int8_t fxe = cos(angle) * (CLOCKSCR_CIRCRAD + (CLOCKSCR_TIMEMKHR / 2));
        /* full 'y' tick end length */
        int8_t fye = sin(angle) * (CLOCKSCR_CIRCRAD + (CLOCKSCR_TIMEMKHR / 2));

        /* plot the line. remember this is with respect to the origin so we
         * need to correct for that by adding LCD_WIDTH / 2 or the ht.
         */

        gl_gfx_Line(disp,
                fxs + (LCD_WIDTH / 2),
                fys + (LCD_HEIGHT / 2),
                fxe + (LCD_WIDTH / 2),
                fye + (LCD_HEIGHT / 2), RED);
    }
}


void lcd_clock_draw_hand(gl_display_t *disp, uint16_t x, uint16_t n, uint8_t l,
        gl_word_t color) {
    /* x = current (minute/second etc., n = total, l = length of hand */

    /* offset to center of clock */
    double angle = ((double) x * M_PI * 2 / n) - M_PI_2;

    /* full 'x' tick end length */
    int8_t fxe = cos(angle) * l;
    /* full 'y' tick end length */
    int8_t fye = sin(angle) * l;

    gl_gfx_Line(disp,
            0 + (LCD_WIDTH / 2),
            0 + (LCD_HEIGHT / 2),
            fxe + (LCD_WIDTH / 2),
            fye + (LCD_HEIGHT / 2), color);
}



