#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

// #include <time.h>
// #include <sys/types.h>
// #include <sys/stat.h>
// #include <fcntl.h>
// #include <termios.h>

#include "freertos/FreeRTOS.h"

#include "Goldelox_Types4D.h"			// defines data types used by the 4D Routines
#include "Goldelox_const4D.h"			// defines for 4dgl constants, generated by conversion of 4DGL constants to target language
#include "gl_error.h"

#define   Err4D_OK              0
#define   Err4D_Timeout         1
#define   Err4D_NAK		2               // other than ACK received


// 4D Global variables
char *Error4DText[] = {"OK", "Timeout", "NAK", "Length", "Invalid"} ;
int    cPort;                                   // comp port handle, used by Intrinsic routines
int    Error4D ;  				// Error indicator,  used and set by Intrinsic routines
unsigned char Error4D_Inv ;                     // Error byte returned from com port, onl set if error = Err_Invalid
int Error_Abort4D ;                             // if true routines will abort when detecting an error
int TimeLimit4D ;                               // time limit in ms for total serial command duration, 2000 (2 seconds) should be adequate for most commands
                                                // assuming a reasonable baud rate AND low latency AND 0 for the Serial Delay Parameter
                                                // temporary increase might be required for very long (bitmap write, large image file opens)
int(*Callback4D) (int, unsigned char) ;         // or indeterminate (eg file_exec, file_run, file_callFunction)  commands

/*
 * Start of 4D Intrinsic Routines
*/

void WriteBytes(gl_serif_t *serif, unsigned char *psOutput, int nCount)
{
    int iOut;
    // Quick return if no device
    if (cPort < 0)
    {
        Error4D = Err4D_OK;
        return;
    }
    // iOut = write(cPort, psOutput, nCount);
    iOut = serif->write_tx(psOutput, nCount);
    serif->flush_tx(TimeLimit4D);
    if (iOut < 0)
    {
        printf("goldelox: write error \n");
    }
    if (iOut != nCount)
        printf("goldelox: Write incomplete!\n");
    return;
}

void WriteChars(gl_serif_t *serif, unsigned char *psOutput)
{
    // Include NULL in output
    WriteBytes(serif, psOutput, strlen((char *)psOutput) + 1);
    return;
}

void WriteWords(gl_serif_t *serif, gl_word_t * Source, int Size)
{
 	gl_word_t wk ;
	int i ;
	for (i = 0; i < Size; i++)
	{
		wk = *Source++ ;
		wk = (wk >> 8) + (wk << 8) ;
		WriteBytes(serif, (unsigned char *)&wk, 2);
	}
}

// Return system time in ms
// GetTickCount not used so removed

// read string from the serial port
// return code:
//   >= 0 = number of characters read
//   -1 = read failed
int ReadSerPort(gl_serif_t *serif, unsigned char *psData, int iMax)
{
    int iIn, iLeft, iIdx;

    // Quick return if no device
    if (!serif)
    {
        Error4D = Err4D_OK;
        *psData = '\0';
        return iMax;
    }
    iIdx = 0;
    iLeft = iMax;
    time_t curTime = time(NULL);
    while (iLeft > 0)
    {
        // iIn = read(cPort, &psData[iIdx], iLeft);
        iIn = serif->read_rx(&psData[iIdx], iLeft, TimeLimit4D);
        if (time(NULL) - curTime > 2) {
            printf("goldelox: timeout \n");
            return -1;
        }
        if (iIn < 0)
        {
            printf("goldelox: read error \n");
            return -1;
        }
        // Anything?
        if (iIn > 0)
        {
            // Calc remaining
            iLeft -= iIn;
            iIdx += iIn;
        }
        // Keep reading
    }

    return iMax;
}

void getbytes(gl_serif_t *serif, unsigned char *data, int size)
{
 	int readc;
	readc = ReadSerPort(serif, data, size) ;
	if ((readc != size)
	    && (Callback4D != NULL) )
	{
		Error4D = Err4D_Timeout ;
		Callback4D(Error4D, Error4D_Inv) ;
 	}
}

void GetAck(gl_serif_t *serif)
{
	int readc;
	unsigned char readx ;
	Error4D = Err4D_OK ;
    // Quick return if no device
    if (cPort < 0)
    {
        Error4D = Err4D_OK;
        return;
    }
   	readc = ReadSerPort(serif, &readx, 1) ;

	if (readc != 1)
    {
		Error4D = Err4D_Timeout ;
        if (Callback4D != NULL)
            Callback4D(Error4D, Error4D_Inv) ;
    }
	else if (readx != 6)
	{
	   	Error4D     = Err4D_NAK ;
	   	Error4D_Inv = readx ;
		if (Callback4D != NULL)
	 		Callback4D(Error4D, Error4D_Inv) ;
	}

    return;
}


gl_word_t GetWord(gl_serif_t *serif)
{
 	int readc;
 	unsigned char readx[2] ;

 	if (Error4D != Err4D_OK)
 		return 0 ;

    readc = ReadSerPort(serif, &readx[0], 2) ;

	if (readc != 2)
	{
		Error4D  = Err4D_Timeout ;
		if (Callback4D != NULL)
	 		return Callback4D(Error4D, Error4D_Inv) ;
		return -Error4D ;
	}
	else
		return readx[0] << 8 | readx[1] ;
}

void getString(gl_serif_t *serif, unsigned char *outStr, int strLen)
{
 	int readc;

 	if (Error4D != Err4D_OK)
	{
		outStr[0] = '\0' ;
 		return ;
	}

	readc = ReadSerPort(serif, outStr, strLen) ;

	if (readc != strLen)
	{
		Error4D  = Err4D_Timeout ;
		if (Callback4D != NULL)
	 		Callback4D(Error4D, Error4D_Inv) ;
	}

    // Append EOS
	outStr[readc] = '\0' ;

	return;
}

gl_word_t GetAckResp(gl_serif_t *serif)
{
	GetAck(serif);
	return GetWord(serif);
}

gl_word_t WaitForAck(gl_serif_t *serif)
{
    int saveTimeout = TimeLimit4D;
    void *saveCB = Callback4D;

    // check once per minute
    Callback4D = NULL;
    TimeLimit4D = 60 * 1000;
    do
    {
        GetAck(serif);
    } while (Error4D != Err4D_OK);

    // Restore callback/timeout saves
    TimeLimit4D = saveTimeout;
    Callback4D = saveCB;

    return GetWord(serif);
}
gl_word_t GetAckRes2Words(gl_serif_t *serif, gl_word_t * word1, gl_word_t * word2)
{
	int Result ;
	GetAck(serif) ;
	Result = GetWord(serif) ;
	*word1 = GetWord(serif) ;
	*word2 = GetWord(serif) ;
	return Result ;
}

void GetAck2Words(gl_serif_t *serif, gl_word_t * word1, gl_word_t * word2)
{
	GetAck(serif) ;
	*word1 = GetWord(serif) ;
	*word2 = GetWord(serif) ;
}

gl_word_t GetAckResSector(gl_serif_t *serif, gl_sector_t Sector)
{
	int Result;
	GetAck(serif) ;
	Result = GetWord(serif) ;
	getbytes(serif, Sector, 512) ;
	return Result ;
}

gl_word_t GetAckResStr(gl_serif_t *serif, unsigned char * OutStr)
{
	int Result ;
	GetAck(serif) ;
	Result = GetWord(serif) ;
	getString(serif, OutStr, Result) ;
	return Result ;
}

gl_word_t GetAckResData(gl_serif_t *serif, gl_byte_array_t OutData, gl_word_t size)
{
	int Result ;
	GetAck(serif) ;
	Result = GetWord(serif) ;
	getbytes(serif, OutData, size) ;
	return Result ;
}

/*
 * End Of Intrinsic 4DRoutines here
*/

/*
 * Starts of 4D Compound Routines
*/

gl_word_t gl_charheight(gl_display_t *display, unsigned char TestChar)
{
  unsigned char  towrite[3] ;
  towrite[0]= F_charheight >> 8 ;
  towrite[1]= F_charheight ;
  towrite[2]= TestChar;
  WriteBytes(display->serif, towrite, 3) ;
  return GetAckResp(display->serif);
}

gl_word_t gl_charwidth(gl_display_t *display, unsigned char TestChar)
{
  unsigned char  towrite[3] ;
  towrite[0]= F_charwidth >> 8 ;
  towrite[1]= F_charwidth ;
  towrite[2]= TestChar;
  WriteBytes(display->serif, towrite, 3) ;
  return GetAckResp(display->serif) ;
}

void gl_gfx_BGcolour(gl_display_t *display, gl_word_t Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_BGcolour >> 8 ;
  towrite[1]= F_gfx_BGcolour & 0xFF;
  towrite[2]= Color >> 8;
  towrite[3]= Color;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif);
}

void gl_gfx_ChangeColour(gl_display_t *display, gl_word_t OldColor,
        gl_word_t NewColor)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_gfx_ChangeColour >> 8 ;
  towrite[1]= F_gfx_ChangeColour ;
  towrite[2]= OldColor >> 8 ;
  towrite[3]= OldColor ;
  towrite[4]= NewColor >> 8 ;
  towrite[5]= NewColor ;
  WriteBytes(display->serif, towrite, 6);
  GetAck(display->serif);
}

void gl_gfx_Circle(gl_display_t *display, gl_word_t  X, gl_word_t  Y,
        gl_word_t  Radius, gl_word_t  Color)
{
  unsigned char  towrite[10] ;
  towrite[0]= F_gfx_Circle >> 8 ;
  towrite[1]= F_gfx_Circle ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Radius >> 8 ;
  towrite[7]= Radius ;
  towrite[8]= Color >> 8 ;
  towrite[9]= Color ;
  WriteBytes(display->serif, towrite, 10) ;
  GetAck(display->serif);
}

void gl_gfx_CircleFilled(gl_display_t *display, gl_word_t  X, gl_word_t  Y,
        gl_word_t  Radius, gl_word_t  Color)
{
  unsigned char  towrite[10] ;
  towrite[0]= F_gfx_CircleFilled >> 8 ;
  towrite[1]= F_gfx_CircleFilled ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Radius >> 8 ;
  towrite[7]= Radius ;
  towrite[8]= Color >> 8 ;
  towrite[9]= Color ;
  WriteBytes(display->serif, towrite, 10) ;
  GetAck(display->serif) ;
}

void gl_gfx_Clipping(gl_display_t *display, gl_word_t OnOff)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_Clipping >> 8 ;
  towrite[1]= F_gfx_Clipping & 0xFF;
  towrite[2]= OnOff >> 8 ;
  towrite[3]= OnOff ;
  WriteBytes(display->serif, towrite, 4);
  GetAck(display->serif);
}

void gl_gfx_ClipWindow(gl_display_t *display, gl_word_t X1, gl_word_t Y1,
        gl_word_t  X2, gl_word_t  Y2)
{
  unsigned char  towrite[10];

  towrite[0]= F_gfx_ClipWindow >> 8;
  towrite[1]= F_gfx_ClipWindow;
  towrite[2]= X1 >> 8;
  towrite[3]= X1;
  towrite[4]= Y1 >> 8;
  towrite[5]= Y1;
  towrite[6]= X2 >> 8;
  towrite[7]= X2;
  towrite[8]= Y2 >> 8;
  towrite[9]= Y2;
  WriteBytes(display->serif, towrite, 10);
  GetAck(display->serif);
}

void gl_gfx_Cls(gl_display_t *display)
{
  unsigned char  towrite[2];

  towrite[0]= F_gfx_Cls >> 8;
  towrite[1]= F_gfx_Cls;
  WriteBytes(display->serif, towrite, 2);
  GetAck(display->serif);
}

void gl_gfx_Contrast(gl_display_t *display, gl_word_t  Contrast)
{
  unsigned char  towrite[4] ;
    towrite[0]= F_gfx_Contrast >> 8 ;
  towrite[1]= F_gfx_Contrast & 0xFF;
  towrite[2]= Contrast >> 8 ;
  towrite[3]= Contrast ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_gfx_FrameDelay(gl_display_t *display, gl_word_t  Msec)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_FrameDelay >> 8 ;
  towrite[1]= F_gfx_FrameDelay & 0xFF;
  towrite[2]= Msec >> 8 ;
  towrite[3]= Msec ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

gl_word_t gl_gfx_GetPixel(gl_display_t *display, gl_word_t X, gl_word_t Y)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_gfx_GetPixel >> 8 ;
  towrite[1]= F_gfx_GetPixel ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(display->serif, towrite, 6);
  return GetAckResp(display->serif);
}

void gl_gfx_Line(gl_display_t *display, gl_word_t  X1, gl_word_t  Y1, gl_word_t  X2, gl_word_t  Y2, gl_word_t  Color)
{
  unsigned char  towrite[12] ;
  towrite[0]= F_gfx_Line >> 8 ;
  towrite[1]= F_gfx_Line ;
  towrite[2]= X1 >> 8 ;
  towrite[3]= X1 ;
  towrite[4]= Y1 >> 8 ;
  towrite[5]= Y1 ;
  towrite[6]= X2 >> 8 ;
  towrite[7]= X2 ;
  towrite[8]= Y2 >> 8 ;
  towrite[9]= Y2 ;
  towrite[10]= Color >> 8 ;
  towrite[11]= Color ;
  WriteBytes(display->serif, towrite, 12) ;
  GetAck(display->serif) ;
}

void gl_gfx_LinePattern(gl_display_t *display, gl_word_t  Pattern)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_LinePattern >> 8 ;
  towrite[1]= F_gfx_LinePattern & 0xFF;
  towrite[2]= Pattern >> 8 ;
  towrite[3]= Pattern ;
  WriteBytes(display->serif, towrite, 4);
  GetAck(display->serif);
}

void gl_gfx_LineTo(gl_display_t *display, gl_word_t  X, gl_word_t  Y)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_gfx_LineTo >> 8 ;
  towrite[1]= F_gfx_LineTo ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_gfx_MoveTo(gl_display_t *display, gl_word_t  X, gl_word_t  Y)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_gfx_MoveTo >> 8 ;
  towrite[1]= F_gfx_MoveTo ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(display->serif, towrite, 6);
  GetAck(display->serif);
}

gl_word_t gl_gfx_Orbit(gl_display_t *display, gl_word_t  Angle, gl_word_t  Distance, gl_word_t *  Xdest, gl_word_t *  Ydest)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_gfx_Orbit >> 8 ;
  towrite[1]= F_gfx_Orbit ;
  towrite[2]= Angle >> 8 ;
  towrite[3]= Angle ;
  towrite[4]= Distance >> 8 ;
  towrite[5]= Distance ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck2Words(display->serif, Xdest,Ydest) ;
  return 0 ;
}

void gl_gfx_OutlineColour(gl_display_t *display, gl_word_t Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_OutlineColour >> 8 ;
  towrite[1]= F_gfx_OutlineColour & 0xFF;
  towrite[2]= Color >> 8 ;
  towrite[3]= Color ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_gfx_Polygon(gl_display_t *display, gl_word_t  n, gl_word_array_t  Xvalues, gl_word_array_t  Yvalues, gl_word_t  Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_Polygon >> 8 ;
  towrite[1]= F_gfx_Polygon ;
  towrite[2]= n >> 8 ;
  towrite[3]= n ;
  WriteBytes(display->serif, towrite, 4) ;
  WriteWords(display->serif,Xvalues, n) ;
  WriteWords(display->serif,Yvalues, n) ;
  towrite[0]= Color >> 8 ;
  towrite[1]= Color ;
  WriteBytes(display->serif,towrite, 2) ;
  GetAck(display->serif) ;
}

void gl_gfx_Polyline(gl_display_t *display, gl_word_t  n, gl_word_array_t  Xvalues, gl_word_array_t  Yvalues, gl_word_t  Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_Polyline >> 8 ;
  towrite[1]= F_gfx_Polyline ;
  towrite[2]= n >> 8 ;
  towrite[3]= n ;
  WriteBytes(display->serif, towrite, 4) ;
  WriteWords(display->serif, Xvalues, n) ;
  WriteWords(display->serif, Yvalues, n) ;
  towrite[0]= Color >> 8 ;
  towrite[1]= Color ;
  WriteBytes(display->serif, towrite, 2) ;
  GetAck(display->serif) ;
}

void gl_gfx_PutPixel(gl_display_t *display, gl_word_t  X, gl_word_t  Y, gl_word_t  Color)
{
  unsigned char  towrite[8] ;
  towrite[0]= F_gfx_PutPixel >> 8 ;
  towrite[1]= F_gfx_PutPixel ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Color >> 8 ;
  towrite[7]= Color ;
  WriteBytes(display->serif, towrite, 8) ;
  GetAck(display->serif) ;
}

void gl_gfx_Rectangle(gl_display_t *display, gl_word_t  X1, gl_word_t  Y1, gl_word_t  X2, gl_word_t  Y2, gl_word_t  Color)
{
  unsigned char  towrite[12] ;
  towrite[0]= F_gfx_Rectangle >> 8 ;
  towrite[1]= F_gfx_Rectangle ;
  towrite[2]= X1 >> 8 ;
  towrite[3]= X1 ;
  towrite[4]= Y1 >> 8 ;
  towrite[5]= Y1 ;
  towrite[6]= X2 >> 8 ;
  towrite[7]= X2 ;
  towrite[8]= Y2 >> 8 ;
  towrite[9]= Y2 ;
  towrite[10]= Color >> 8 ;
  towrite[11]= Color ;
  WriteBytes(display->serif, towrite, 12) ;
  GetAck(display->serif) ;
}

void gl_gfx_RectangleFilled(gl_display_t *display, gl_word_t  X1, gl_word_t  Y1, gl_word_t  X2, gl_word_t  Y2, gl_word_t  Color)
{
  unsigned char  towrite[12] ;
  towrite[0]= F_gfx_RectangleFilled >> 8 ;
  towrite[1]= F_gfx_RectangleFilled ;
  towrite[2]= X1 >> 8 ;
  towrite[3]= X1 ;
  towrite[4]= Y1 >> 8 ;
  towrite[5]= Y1 ;
  towrite[6]= X2 >> 8 ;
  towrite[7]= X2 ;
  towrite[8]= Y2 >> 8 ;
  towrite[9]= Y2 ;
  towrite[10]= Color >> 8 ;
  towrite[11]= Color ;
  WriteBytes(display->serif, towrite, 12) ;
  GetAck(display->serif) ;
}

void gl_gfx_ScreenMode(gl_display_t *display, gl_word_t  ScreenMode)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_ScreenMode >> 8 ;
  towrite[1]= F_gfx_ScreenMode & 0xFF;
  towrite[2]= ScreenMode >> 8 ;
  towrite[3]= ScreenMode ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_gfx_Set(gl_display_t *display, gl_word_t  Func, gl_word_t  Value)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_gfx_Set >> 8 ;
  towrite[1]= F_gfx_Set ;
  towrite[2]= Func >> 8 ;
  towrite[3]= Func ;
  towrite[4]= Value >> 8 ;
  towrite[5]= Value ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_gfx_SetClipRegion(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_gfx_SetClipRegion >> 8 ;
  towrite[1]= F_gfx_SetClipRegion ;
  WriteBytes(display->serif, towrite, 2) ;
  GetAck(display->serif) ;
}

void gl_gfx_Transparency(gl_display_t *display, gl_word_t  OnOff)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_Transparency >> 8 ;
  towrite[1]= F_gfx_Transparency & 0xFF;
  towrite[2]= OnOff >> 8 ;
  towrite[3]= OnOff ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_gfx_TransparentColour(gl_display_t *display, gl_word_t  Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_gfx_TransparentColour >> 8 ;
  towrite[1]= F_gfx_TransparentColour & 0xFF;
  towrite[2]= Color >> 8 ;
  towrite[3]= Color ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_gfx_Triangle(gl_display_t *display, gl_word_t  X1, gl_word_t  Y1, gl_word_t  X2, gl_word_t  Y2, gl_word_t  X3, gl_word_t  Y3, gl_word_t  Color)
{
  unsigned char  towrite[16] ;
  towrite[0]= F_gfx_Triangle >> 8 ;
  towrite[1]= F_gfx_Triangle ;
  towrite[2]= X1 >> 8 ;
  towrite[3]= X1 ;
  towrite[4]= Y1 >> 8 ;
  towrite[5]= Y1 ;
  towrite[6]= X2 >> 8 ;
  towrite[7]= X2 ;
  towrite[8]= Y2 >> 8 ;
  towrite[9]= Y2 ;
  towrite[10]= X3 >> 8 ;
  towrite[11]= X3 ;
  towrite[12]= Y3 >> 8 ;
  towrite[13]= Y3 ;
  towrite[14]= Color >> 8 ;
  towrite[15]= Color ;
  WriteBytes(display->serif, towrite, 16) ;
  GetAck(display->serif) ;
}

gl_word_t gl_media_Flush(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_media_Flush >> 8 ;
  towrite[1]= F_media_Flush ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

void gl_media_Image(gl_display_t *display, gl_word_t  X, gl_word_t  Y)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_media_Image >> 8 ;
  towrite[1]= F_media_Image ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

gl_word_t gl_media_Init(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_media_Init >> 8 ;
  towrite[1]= F_media_Init ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

gl_word_t gl_media_ReadByte(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_media_ReadByte >> 8 ;
  towrite[1]= F_media_ReadByte ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

gl_word_t gl_media_ReadWord(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_media_ReadWord >> 8 ;
  towrite[1]= F_media_ReadWord ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

void gl_media_SetAdd(gl_display_t *display, gl_word_t  HiWord, gl_word_t  LoWord)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_media_SetAdd >> 8 ;
  towrite[1]= F_media_SetAdd ;
  towrite[2]= HiWord >> 8 ;
  towrite[3]= HiWord ;
  towrite[4]= LoWord >> 8 ;
  towrite[5]= LoWord ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_media_SetSector(gl_display_t *display, gl_word_t  HiWord, gl_word_t  LoWord)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_media_SetSector >> 8 ;
  towrite[1]= F_media_SetSector ;
  towrite[2]= HiWord >> 8 ;
  towrite[3]= HiWord ;
  towrite[4]= LoWord >> 8 ;
  towrite[5]= LoWord ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_media_Video(gl_display_t *display, gl_word_t  X, gl_word_t  Y)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_media_Video >> 8 ;
  towrite[1]= F_media_Video ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_media_VideoFrame(gl_display_t *display, gl_word_t  X, gl_word_t  Y, gl_word_t  Framenumber)
{
  unsigned char  towrite[8] ;
  towrite[0]= F_media_VideoFrame >> 8 ;
  towrite[1]= F_media_VideoFrame ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Framenumber >> 8 ;
  towrite[7]= Framenumber ;
  WriteBytes(display->serif, towrite, 8) ;
  GetAck(display->serif) ;
}

gl_word_t gl_media_WriteByte(gl_display_t *display, gl_word_t  Byte)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_media_WriteByte >> 8 ;
  towrite[1]= F_media_WriteByte ;
  towrite[2]= Byte >> 8 ;
  towrite[3]= Byte ;
  WriteBytes(display->serif, towrite, 4) ;
  return GetAckResp(display->serif) ;
}

gl_word_t gl_media_WriteWord(gl_display_t *display, gl_word_t  Word)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_media_WriteWord >> 8 ;
  towrite[1]= F_media_WriteWord ;
  towrite[2]= Word >> 8 ;
  towrite[3]= Word ;
  WriteBytes(display->serif, towrite, 4) ;
  return GetAckResp(display->serif) ;
}

void gl_putCH(gl_display_t *display, gl_word_t  WordChar)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_putCH >> 8 ;
  towrite[1]= F_putCH ;
  towrite[2]= WordChar >> 8 ;
  towrite[3]= WordChar ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_putstr(gl_display_t *display, unsigned char *  InString)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_putstr >> 8 ;
  towrite[1]= F_putstr ;
  WriteBytes(display->serif, towrite, 2) ;
  WriteChars(display->serif, InString) ;
  GetAck(display->serif) ;
}

void gl_txt_Attributes(gl_display_t *display, gl_word_t  Attribs)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Attributes >> 8 ;
  towrite[1]= F_txt_Attributes & 0xFF;
  towrite[2]= Attribs >> 8 ;
  towrite[3]= Attribs ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_BGcolour(gl_display_t *display, gl_word_t  Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_BGcolour >> 8 ;
  towrite[1]= F_txt_BGcolour & 0xFF;
  towrite[2]= Color >> 8 ;
  towrite[3]= Color ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Bold(gl_display_t *display, gl_word_t  Bold)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Bold >> 8 ;
  towrite[1]= F_txt_Bold & 0xFF;
  towrite[2]= Bold >> 8 ;
  towrite[3]= Bold ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_FGcolour(gl_display_t *display, gl_word_t  Color)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_FGcolour >> 8 ;
  towrite[1]= F_txt_FGcolour & 0xFF;
  towrite[2]= Color >> 8 ;
  towrite[3]= Color ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_FontID(gl_display_t *display, gl_word_t  FontNumber)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_FontID >> 8 ;
  towrite[1]= F_txt_FontID & 0xFF;
  towrite[2]= FontNumber >> 8 ;
  towrite[3]= FontNumber ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Height(gl_display_t *display, gl_word_t  Multiplier)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Height >> 8 ;
  towrite[1]= F_txt_Height & 0xFF;
  towrite[2]= Multiplier >> 8 ;
  towrite[3]= Multiplier ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Inverse(gl_display_t *display, gl_word_t  Inverse)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Inverse >> 8 ;
  towrite[1]= F_txt_Inverse & 0xFF;
  towrite[2]= Inverse >> 8 ;
  towrite[3]= Inverse ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Italic(gl_display_t *display, gl_word_t  Italic)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Italic >> 8 ;
  towrite[1]= F_txt_Italic & 0xFF;
  towrite[2]= Italic >> 8 ;
  towrite[3]= Italic ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_MoveCursor(gl_display_t *display, gl_word_t  Line, gl_word_t  Column)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_txt_MoveCursor >> 8 ;
  towrite[1]= F_txt_MoveCursor ;
  towrite[2]= Line >> 8 ;
  towrite[3]= Line ;
  towrite[4]= Column >> 8 ;
  towrite[5]= Column ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_txt_Opacity(gl_display_t *display, gl_word_t  TransparentOpaque)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Opacity >> 8 ;
  towrite[1]= F_txt_Opacity & 0xFF;
  towrite[2]= TransparentOpaque >> 8 ;
  towrite[3]= TransparentOpaque ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Set(gl_display_t *display, gl_word_t  Func, gl_word_t  Value)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_txt_Set >> 8 ;
  towrite[1]= F_txt_Set ;
  towrite[2]= Func >> 8 ;
  towrite[3]= Func ;
  towrite[4]= Value >> 8 ;
  towrite[5]= Value ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

void gl_txt_Underline(gl_display_t *display, gl_word_t  Underline)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Underline >> 8 ;
  towrite[1]= F_txt_Underline & 0xFF;
  towrite[2]= Underline >> 8 ;
  towrite[3]= Underline ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Width(gl_display_t *display, gl_word_t  Multiplier)
{
  unsigned char  towrite[4] ;

  towrite[0]= F_txt_Width >> 8 ;
  towrite[1]= F_txt_Width & 0xFF;
  towrite[2]= Multiplier >> 8 ;
  towrite[3]= Multiplier ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Xgap(gl_display_t *display, gl_word_t  Pixels)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Xgap >> 8 ;
  towrite[1]= F_txt_Xgap & 0xFF;
  towrite[2]= Pixels >> 8 ;
  towrite[3]= Pixels ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_txt_Ygap(gl_display_t *display, gl_word_t  Pixels)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_txt_Ygap >> 8 ;
  towrite[1]= F_txt_Ygap & 0xFF;
  towrite[2]= Pixels >> 8 ;
  towrite[3]= Pixels ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_BeeP(gl_display_t *display, gl_word_t  Note, gl_word_t  Duration)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_BeeP >> 8 ;
  towrite[1]= F_BeeP ;
  towrite[2]= Note >> 8 ;
  towrite[3]= Note ;
  towrite[4]= Duration >> 8 ;
  towrite[5]= Duration ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

gl_word_t gl_sys_GetModel(gl_display_t *display, unsigned char *  ModelStr)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_sys_GetModel >> 8 ;
  towrite[1]= F_sys_GetModel ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResStr(display->serif, ModelStr) ;
}

gl_word_t gl_sys_GetVersion(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_sys_GetVersion >> 8 ;
  towrite[1]= F_sys_GetVersion ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

gl_word_t gl_sys_GetPmmC(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_sys_GetPmmC >> 8 ;
  towrite[1]= F_sys_GetPmmC ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

void gl_blitComtoDisplay(gl_display_t *display, gl_word_t  X, gl_word_t  Y, gl_word_t  Width, gl_word_t  Height, gl_byte_array_t  Pixels)
{
  unsigned char  towrite[10] ;
  towrite[0]= F_blitComtoDisplay >> 8 ;
  towrite[1]= F_blitComtoDisplay ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Width >> 8 ;
  towrite[7]= Width ;
  towrite[8]= Height >> 8 ;
  towrite[9]= Height ;
  WriteBytes(display->serif, towrite, 10) ;
  vTaskDelay(pdMS_TO_TICKS(1));
  WriteBytes(display->serif, Pixels, Width*Height*2) ;
  GetAck(display->serif) ;
}

void gl_setbaudWait(gl_display_t *display, gl_word_t  Newrate,
        void (*cb)())
{
  unsigned char  towrite[4] ;
  towrite[0]= F_setbaudWait >> 8 ;
  towrite[1]= F_setbaudWait ;
  towrite[2]= Newrate >> 8 ;
  towrite[3]= Newrate ;
  WriteBytes(display->serif, towrite, 4) ;
  //SetThisBaudrate(Newrate) ; // change this systems baud rate to match new display rate, ACK is 100ms away
  cb();
  GetAck(display->serif) ;
}

gl_word_t gl_peekW(gl_display_t *display, gl_word_t  Address)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_peekW >> 8 ;
  towrite[1]= F_peekW ;
  towrite[2]= Address >> 8 ;
  towrite[3]= Address ;
  WriteBytes(display->serif, towrite, 4) ;
  return GetAckResp(display->serif) ;
}

void gl_pokeW(gl_display_t *display, gl_word_t  Address, gl_word_t  WordValue)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_pokeW >> 8 ;
  towrite[1]= F_pokeW ;
  towrite[2]= Address >> 8 ;
  towrite[3]= Address ;
  towrite[4]= WordValue >> 8 ;
  towrite[5]= WordValue ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

gl_word_t gl_peekB(gl_display_t *display, gl_word_t  Address)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_peekB >> 8 ;
  towrite[1]= F_peekB ;
  towrite[2]= Address >> 8 ;
  towrite[3]= Address ;
  WriteBytes(display->serif, towrite, 4) ;
  return GetAckResp(display->serif) ;
}

void gl_pokeB(gl_display_t *display, gl_word_t  Address, gl_word_t  ByteValue)
{
  unsigned char  towrite[6] ;
  towrite[0]= F_pokeB >> 8 ;
  towrite[1]= F_pokeB ;
  towrite[2]= Address >> 8 ;
  towrite[3]= Address ;
  towrite[4]= ByteValue >> 8 ;
  towrite[5]= ByteValue ;
  WriteBytes(display->serif, towrite, 6) ;
  GetAck(display->serif) ;
}

gl_word_t gl_joystick(gl_display_t *display)
{
  unsigned char  towrite[2] ;
  towrite[0]= F_joystick >> 8 ;
  towrite[1]= F_joystick ;
  WriteBytes(display->serif, towrite, 2) ;
  return GetAckResp(display->serif) ;
}

void gl_SSTimeout(gl_display_t *display, gl_word_t  Seconds)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_SSTimeout >> 8 ;
  towrite[1]= F_SSTimeout ;
  towrite[2]= Seconds >> 8 ;
  towrite[3]= Seconds ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_SSSpeed(gl_display_t *display, gl_word_t  Speed)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_SSSpeed >> 8 ;
  towrite[1]= F_SSSpeed ;
  towrite[2]= Speed >> 8 ;
  towrite[3]= Speed ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

void gl_SSMode(gl_display_t *display, gl_word_t  Parm)
{
  unsigned char  towrite[4] ;
  towrite[0]= F_SSMode >> 8 ;
  towrite[1]= F_SSMode ;
  towrite[2]= Parm >> 8 ;
  towrite[3]= Parm ;
  WriteBytes(display->serif, towrite, 4) ;
  GetAck(display->serif) ;
}

/*
void setbaudWait(gl_word_t  Newrate)
{
  unsigned char  towrite[4] ;

  towrite[0]= F_setbaudWait >> 8 ;
  towrite[1]= F_setbaudWait ;
  towrite[2]= Newrate >> 8 ;
  towrite[3]= Newrate ;
  WriteBytes(towrite, 4) ;
  //SetThisBaudrate(Newrate) ; // change this systems baud rate to match new display rate, ACK is 100ms away
  GetAck() ;
}
*/

/*
 * Conpound 4D Routines Ends here
*/


gl_err_t gl_init(gl_display_t *disp, gl_serif_t *serif) {
    int k, tSave;
    unsigned char ch;
    //gl_word_t    nBaud;

    TimeLimit4D = 2000;

    if (!disp || !serif)
        return GL_ENULL;

    disp->serif = serif;



    tSave = TimeLimit4D;
    TimeLimit4D = 500;
    for (k = 0 ; k < 10 ; k++)
    {
        ch = 'X';
        disp->serif->write_tx((void *)&ch, 1);
        disp->serif->flush_tx(TimeLimit4D);
        if (ReadSerPort(disp->serif, &ch, 1) == -1)
            break;
        if (ch == 0x15)
            break ;
    }
    TimeLimit4D = tSave;

    disp->serif->flush_rx();
    return 0;
}

// int OpenComm(char *sDeviceName, int newrate)
// {
//     struct termios new_port_settings;
//     //gl_word_t    nBaud;
//     int k, ch, tSave, baudr;
//     
// switch(newrate)
//   {
//     case      50 : baudr = B50;         break;
//     case      75 : baudr = B75;         break;
//     case     110 : baudr = B110;        break;
//     case     134 : baudr = B134;        break;
//     case     150 : baudr = B150;        break;
//     case     200 : baudr = B200;        break;
//     case     300 : baudr = B300;        break;
//     case     600 : baudr = B600;        break;
//     case    1200 : baudr = B1200;       break;
//     case    1800 : baudr = B1800;       break;
//     case    2400 : baudr = B2400;       break;
//     case    4800 : baudr = B4800;       break;
//     case    9600 : baudr = B9600;       break;
//     case   19200 : baudr = B19200;      break;
//     case   38400 : baudr = B38400;      break;
//     case   57600 : baudr = B57600;      break;
//     case  115200 : baudr = B115200;     break;
//     case  230400 : baudr = B230400;     break;
//     case  460800 : baudr = B460800;     break;
//     case  500000 : baudr = B500000;     break;
//     case  576000 : baudr = B576000;     break;
//     case  921600 : baudr = B921600;     break;
//     case 1000000 : baudr = B1000000;    break;
//     default      : printf("invalid baudrate\n");
//                    return(1);
//                    break;
// }
//     cPort = open(sDeviceName, O_RDWR | O_NOCTTY | O_NDELAY);
// // Current config
//     tcgetattr(cPort, &new_port_settings);
//     // Set the line to RAW
//     cfmakeraw(&new_port_settings);
//     memset(&new_port_settings, 0, sizeof(new_port_settings));  /* clear the new struct */
//     new_port_settings.c_cflag = baudr | CS8 | CLOCAL | CREAD;
//     new_port_settings.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
//     new_port_settings.c_iflag = IGNPAR;
//     new_port_settings.c_oflag = 0;
//     new_port_settings.c_lflag = 0;
//     new_port_settings.c_cc[VMIN] = 0;      /* block untill n bytes are received */
//     new_port_settings.c_cc[VTIME] = 100;     /* block untill a timer expires (n * 100 mSec.) */
// /*
//     cfsetospeed(&new_port_settings, nBaud);
//     cfsetispeed(&new_port_settings, nBaud);
// */
//     // set new config
//     tcsetattr(cPort, TCSANOW, &new_port_settings);
//     // Set non-blocking
//     fcntl(cPort, F_SETFL, FNDELAY);
//     
//     tSave = TimeLimit4D;
//     TimeLimit4D = 500;
//     for (k = 0 ; k < 10 ; k++)
//     {
//         ch = 'X';
//         write(cPort, (unsigned char *)&ch, 1);
//         tcflush(cPort, TCOFLUSH);
//         ReadSerPort((unsigned char *)&ch, 1);
//         if (ch == 0x15)
//             break ;
//     }
//     TimeLimit4D = tSave;
// 
//     tcflush(cPort, TCIOFLUSH);
//     return 0;
// }
// 
// void CloseComm(void)
// {
//     close(cPort);
//     cPort = -1;
//     Error4D = Err4D_OK;
// 
//     return;
// }

void gl_reset2(gl_display_t *disp) {
    int k, tSave;
    unsigned char ch;
    //gl_word_t    nBaud;

    TimeLimit4D = 2000;

    tSave = TimeLimit4D;
    TimeLimit4D = 500;
    for (k = 0 ; k < 10 ; k++)
    {
        ch = 'X';
        disp->serif->write_tx((void *)&ch, 1);
        disp->serif->flush_tx(TimeLimit4D);
        if (ReadSerPort(disp->serif, &ch, 1) == -1)
            break;
        if (ch == 0x15)
            break ;
    }
    TimeLimit4D = tSave;

    disp->serif->flush_rx();
}


gl_word_t gl_txt_strw(gl_display_t *disp, char *str) {
    gl_word_t pxlw = 0;
    while (*str) {
        pxlw += gl_charwidth(disp, (unsigned char) *str);
        str++;
    }
    return pxlw;
}